SET QUOTED_IDENTIFIER ON

GO
-- ============================================================
-- FAB Seed Script - Idempotent with OUTER APPLY
-- ============================================================

SET NOCOUNT ON;
SET XACT_ABORT ON;

PRINT '============================================================';
PRINT 'FAB Seed Script (Idempotent)';
PRINT 'Started: ' + CONVERT(VARCHAR(30), GETDATE(), 120);
PRINT '============================================================';

BEGIN TRY
BEGIN TRANSACTION;

-- ============================================================
-- 1. CATALOGS
-- ============================================================
PRINT '';
PRINT '=== Step 1: Catalogs ===';

MERGE FAB.catCustomers AS target
USING (VALUES
    ('A M Haire'),
    ('Alfred'),
    ('Aurora'),
    ('Brown Cargo'),
    ('Demountable'),
    ('Demountable Con'),
    ('Dorsey'),
    ('Durabody'),
    ('Evolve Solutions'),
    ('Fleet'),
    ('Fruehauf'),
    ('Great Dane'),
    ('Great Dane "Danville"'),
    ('Hercules'),
    ('Hyundai'),
    ('J Van Body'),
    ('Kidron'),
    ('Lifeline Mobile'),
    ('Manac'),
    ('Marion Body'),
    ('Marshall'),
    ('Mickey'),
    ('Mickey Body'),
    ('Morgan'),
    ('Multivan'),
    ('Pitts'),
    ('Stoughton'),
    ('Strick'),
    ('Supreme'),
    ('U.S. Truck'),
    ('Utilimaster'),
    ('Utility'),
    ('Vanguard'),
    ('Wabash'),
    ('XPO')
) AS source (CustomerName)
ON target.CustomerName = source.CustomerName
WHEN NOT MATCHED THEN INSERT (CustomerName, CreatedBy) VALUES (source.CustomerName, 'seed')
WHEN MATCHED THEN UPDATE SET ModifiedBy = 'seed', ModifiedDate = GETUTCDATE();
PRINT CONCAT('catCustomers: ', @@ROWCOUNT, ' merged');

MERGE FAB.catSections AS target
USING (VALUES
    ('2559'),
    ('2695'),
    ('2716'),
    ('2718'),
    ('2720'),
    ('2744'),
    ('2754'),
    ('2806'),
    ('2807'),
    ('2919'),
    ('2982'),
    ('2990'),
    ('8695')
) AS source (SectionName)
ON target.SectionName = source.SectionName
WHEN NOT MATCHED THEN INSERT (SectionName, CreatedBy) VALUES (source.SectionName, 'seed')
WHEN MATCHED THEN UPDATE SET ModifiedBy = 'seed', ModifiedDate = GETUTCDATE();
PRINT CONCAT('catSections: ', @@ROWCOUNT, ' merged');

MERGE FAB.catGrades AS target
USING (VALUES
    ('1'),
    ('41')
) AS source (GradeName)
ON target.GradeName = source.GradeName
WHEN NOT MATCHED THEN INSERT (GradeName, CreatedBy) VALUES (source.GradeName, 'seed')
WHEN MATCHED THEN UPDATE SET ModifiedBy = 'seed', ModifiedDate = GETUTCDATE();
PRINT CONCAT('catGrades: ', @@ROWCOUNT, ' merged');

-- ============================================================
-- 2. CLIPS
-- ============================================================
PRINT '';
PRINT '=== Step 2: catClips ===';

MERGE FAB.catClips AS target
USING (VALUES
    (3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
    (2.875, 3.25, '2 Hole-0.313', 'W26 / W26L', 'W25'),
    (3.5, 3.5, 'Blank', 'W18', 'W16'),
    (5.0, 4.938, '4 Hole-0.406', 'W5', 'W1'),
    (3.0, 3.938, '4 Hole-0.406', 'W44', 'W4'),
    (3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
    (4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
    (5.197, 4.016, '4 Hole-0.406', 'Flush to Bottom', 'W48'),
    (5.197, 4.016, '5 Hole-0.406', 'Flush to Bottom', 'W48'),
    (2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
    (4.0, 2.5, '4 Hole-0.406', 'W37', 'W4'),
    (3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
    (5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
    (5.0, 3.5, 'Blank', 'W18', 'W1'),
    (4.0, 5.75, 'Blank', 'Flush to Bottom', 'W47'),
    (5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
    (5.0, 3.85, 'Blank', 'W19', 'W17'),
    (5.75, 4.0, 'Blank', 'Flush to Bottom', 'W47'),
    (5.0, 4.0, '4-Drain Holes only', 'Flush to Bottom', 'W1'),
    (5.0, 3.85, 'Blank', 'W19', 'W1'),
    (5.0, 4.0, 'Blank', 'Flush to Bottom', 'W17'),
    (5.0, 4.0, '4-Drain Holes only', 'Flush to Bottom', 'W17'),
    (3.0, 2.5, 'Blank', 'Flush to Bottom', 'W46'),
    (5.0, 4.0, '8 Hole-.0406', 'Flush to Bottom', 'W1'),
    (5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W17'),
    (5.0, 4.0, '4 Drain Holes', 'Flush to Bottom', 'W1'),
    (5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
    (5.0, 4.0, 'Blank', 'Flush to Bottom', NULL),
    (5.0, 4.0, '8 hole-.0406 3" Pattern', 'Flush to Bottom', 'W17'),
    (3.5, 3.0, 'Blank', 'W18', 'W16'),
    (5.0, 4.938, '4 Hole-0.406', NULL, NULL),
    (5.0, 4.0, 'Blank', 'W19', 'W17'),
    (5.0, 4.0, 'Drain Hole', 'Flush to Bottom', 'W17'),
    (5.0, 4.0, 'Blank', 'W19 Flush to Bottom', 'W1'),
    (4.25, 2.75, '4 Hole-0.406', NULL, NULL),
    (5.197, 4.016, '1-Hole', 'Flush to Bottom', 'W48'),
    (5.197, 4.016, 'Blank', 'Flush to Bottom', 'W48'),
    (3.197, 4.016, '4 Hole-0.406', 'Flush to Bottom', 'W48'),
    (3.197, 4.016, '4 Hole-0.406', 'Flush to Bottom', NULL),
    (3.0, 2.25, '4 Hole-0.406', 'W31', 'W30'),
    (3.0, 2.25, '2 Hole-0.406', 'W31', 'W30'),
    (3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
    (3.0, 4.0, '4 Hole-0.406', 'W42', NULL),
    (5.0, 3.5, 'Blank', 'Flush to Bottom', 'W1'),
    (4.25, 2.75, 'Blank', NULL, 'W22'),
    (4.25, 2.75, 'Blank', '`', 'W22'),
    (4.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'XX'),
    (3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
    (3.938, 5.125, 'Blank', NULL, 'W45'),
    (6.0, 2.125, 'Unknown', NULL, NULL),
    (3.0, 3.031, '8 Hole-0.406', 'W7', 'W3'),
    (2.875, 3.25, '2 Hole-0.406', 'W36 / W36L', 'W35'),
    (2.875, 3.25, '2 Hole-0.406', 'W36 / W36L', 'W38'),
    (3.0, 2.75, '4 Hole-0.391', 'W12', 'W13'),
    (3.0, 2.907, '4 Hole-0.391', 'W40', 'W4'),
    (3.938, 2.25, '6 Hole-0.391', 'W12', 'W11'),
    (5.0, 4.25, '4 Hole-0.406', 'W14', 'W49'),
    (4.25, 2.5, '4 Hole-0.406', 'W14', 'W15'),
    (5.0, 3.25, '4 Hole-0.406', 'W41', 'W1'),
    (5.0, 4.25, '2 Hole-0.406', 'W14', 'W49'),
    (3.0, 3.25, '4 Hole-0.406', 'W41', 'W4'),
    (5.0, 3.938, '4 Hole-0.406', 'W5', 'W2'),
    (3.0, 3.0, 'Blank', 'Flush to Bottom', 'W3'),
    (3.0, 3.0, 'Blank', NULL, 'W3'),
    (3.0, 4.0, '4 Hole-0.406', 'W7', 'W4'),
    (5.0, 4.938, '4 Hole-0.534 Alpha', 'W6', 'W1'),
    (5.0, 4.0, '4 Hole-0.534 Alpha', 'W6', 'W2'),
    (4.0, 5.25, 'Blank', NULL, 'W39'),
    (3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
    (3.0, 3.031, '7 Hole-0.406', 'W7', 'W3'),
    (3.0, 3.875, '4 Hole-0.406', 'W7', 'W4'),
    (3.0, 3.031, '7 Hole-0.406', 'Flush to Bottom', 'W4'),
    (4.0, 5.0, '2 Hole', 'Flush to Bottom', 'W50'),
    (3.0, 4.75, '5 Hole-0.438', NULL, NULL),
    (3.0, 4.75, 'Unknown', NULL, NULL),
    (5.0, 3.75, '5 Hole 0.406', 'Split Difference', 'W1')
) AS source (Width, Height, HolePattern, HoleGage, SideGage)
ON target.Width = source.Width 
   AND target.Height = source.Height 
   AND target.HolePattern = source.HolePattern
   AND ISNULL(target.HoleGage, '') = ISNULL(source.HoleGage, '')
   AND ISNULL(target.SideGage, '') = ISNULL(source.SideGage, '')
WHEN NOT MATCHED THEN 
    INSERT (Width, Height, HolePattern, HoleGage, SideGage, CreatedBy) 
    VALUES (source.Width, source.Height, source.HolePattern, source.HoleGage, source.SideGage, 'seed')
WHEN MATCHED THEN
    UPDATE SET ModifiedBy = 'seed', ModifiedDate = GETUTCDATE();
PRINT CONCAT('catClips: ', @@ROWCOUNT, ' merged');

-- ============================================================
-- 3. PRODUCTS
-- ============================================================
PRINT '';
PRINT '=== Step 3: Products ===';

IF OBJECT_ID('tempdb..#ProductStaging') IS NOT NULL DROP TABLE #ProductStaging;

CREATE TABLE #ProductStaging (
    PartNumber VARCHAR(50),
    CustomerName VARCHAR(100),
    SectionName VARCHAR(20),
    GradeName VARCHAR(20)
);

INSERT INTO #ProductStaging VALUES
('3526141C', 'Alfred', '2695', '1'),
('35261410C', 'Alfred', '2695', '1'),
('31098-01', 'A M Haire', '2919', '1'),
('31098', 'A M Haire', '2919', '1'),
('31099', 'A M Haire', '2919', '1'),
('A33166', 'Aurora', '2720', '41'),
('A02000010', 'Aurora', '2695', '41'),
('A02000141', 'Aurora', '2695', '41'),
('A0910076', 'Aurora', '2695', '41'),
('A02000393', 'Aurora', '2695', '41'),
('A02001133', 'Aurora', '2559', '41'),
('ACF51177C', 'Aurora', '2695', '41'),
('A10031775', 'Aurora', '2695', '41'),
('A18454001', 'Aurora', '2982', '41'),
('A3-101BC', 'Aurora', '2695', '41'),
('20071290', 'Aurora', '2559', '41'),
('BCV010002', 'Brown Cargo', '2919', '1'),
('BCV010021', 'Brown Cargo', '2919', '1'),
('BCV010025', 'Brown Cargo', '2919', '1'),
('BCV10025G', 'Brown Cargo', '2919', '1'),
('291995325', 'Brown Cargo', '2919', '1'),
('291910132', 'Brown Cargo', '2919', '1'),
('26959131P', 'Brown Cargo', '2695', '41'),
('26959562P', 'Brown Cargo', '2695', '41'),
('26951015D', 'Durabody', '2695', '41'),
('FCE7662-7', 'Fruehauf', '2695', '41'),
('FCF88914C', 'Fruehauf', '2695', '41'),
('FCF51177C', 'Fruehauf', '2695', '41'),
('SP0019001', 'Fruehauf', '2695', '41'),
('SP0019002', 'Fruehauf', '2695', '41'),
('F03510100', 'Great Dane', '2695', '41'),
('30101506', 'Great Dane', '2695', '41'),
('30101511', 'Great Dane', '2754', '41'),
('30101536', 'Great Dane', '2754', '41'),
('30101511B', 'Great Dane', '2754', '41'),
('30101511Z', 'Great Dane', '2754', '41'),
('30101511M', 'Great Dane', '2754', '41'),
('30101511G', 'Great Dane', '2754', '41'),
('30101514', 'Great Dane', '2720', '41'),
('30101514B', 'Great Dane', '2720', '41'),
('30101514G', 'Great Dane', '2720', '41'),
('30101515', 'Great Dane', '2695', '41'),
('30101515B', 'Great Dane', '2695', '41'),
('30101515G', 'Great Dane', '2695', '41'),
('30101516', 'Great Dane', '2695', '41'),
('30101516B', 'Great Dane', '2695', '41'),
('30101516G', 'Great Dane', '2695', '41'),
('30101517', 'Great Dane', '2716', '41'),
('30101517B', 'Great Dane', '2716', '41'),
('30101517G', 'Great Dane', '2716', '41'),
('30101518', 'Great Dane', '2754', '41'),
('30101518B', 'Great Dane', '2754', '41'),
('30101518G', 'Great Dane', '2754', '41'),
('30101518M', 'Great Dane', '2754', '41'),
('30101518T', 'Great Dane', '2754', '41'),
('30100041', 'Great Dane', '2695', '41'),
('30100228', 'Great Dane', '2695', '41'),
('30100178', 'Great Dane', '2695', '41'),
('30101519', 'Great Dane', '2720', '41'),
('30101519B', 'Great Dane', '2720', '41'),
('30101519G', 'Great Dane', '2720', '41'),
('30101520', 'Great Dane', '2716', '41'),
('30101520B', 'Great Dane', '2716', '41'),
('30101520G', 'Great Dane', '2716', '41'),
('30101523', 'Great Dane', '2559', '41'),
('30101524', 'Great Dane', '2716', '41'),
('30101524G', 'Great Dane', '2716', '41'),
('30101524M', 'Great Dane', '2716', '41'),
('30101526', 'Great Dane', '2754', '41'),
('30101526B', 'Great Dane', '2754', '41'),
('30101527', 'Great Dane', '2716', '41'),
('30106314B', 'Great Dane', '2716', '41'),
('30101534', 'Great Dane', '2716', '41'),
('30101534B', 'Great Dane', '2716', '41'),
('30101534G', 'Great Dane', '2716', '41'),
('30101539', 'Great Dane', '2720', '41'),
('30106221', 'Great Dane', '2695', '41'),
('30106278', 'Great Dane', '2919', '1'),
('30106278B', 'Great Dane', '2919', '1'),
('30106293G', 'Great Dane', '2695', '41'),
('30106262', 'Great Dane', '2695', '41'),
('30106296', 'Great Dane', '2754', '41'),
('30106252', 'Great Dane', '2716', '41'),
('30106298', 'Great Dane', '2754', '41'),
('30106326', 'Great Dane', '2754', '41'),
('30106336', 'Great Dane', '2754', '41'),
('30106346', 'Great Dane', '2754', '41'),
('30106358', 'Great Dane', '2919', '1'),
('30106358B', 'Great Dane', '2919', '1'),
('30110002', 'Great Dane', '2754', '41'),
('30110002D', 'Great Dane "Danville"', '2754', '41'),
('30100042', 'Great Dane', '2695', '41'),
('30100025', 'Great Dane', '2695', '41'),
('30100024', 'Great Dane', '2695', '41'),
('30100024B', 'Great Dane', '2695', '41'),
('30100043', 'Great Dane', '2695', '41'),
('30100044', 'Great Dane', '2695', '41'),
('30100149', 'Great Dane', '2695', '41'),
('30110041', 'Great Dane', '2754', '41'),
('30110041D', 'Great Dane "Danville"', '2754', '41'),
('30110096', 'Great Dane', '2718', '41'),
('30110130', 'Great Dane', '2754', '41'),
('30110132', 'Great Dane', '2754', '41'),
('30110134', 'Great Dane', '2716', '41'),
('30110138', 'Great Dane', '2754', '41'),
('30110106', 'Great Dane', '2695', '41'),
('30133446', 'Great Dane', '2807', '41'),
('33166', 'Great Dane', '2720', '41'),
('33016', 'Great Dane', '2720', '41'),
('40103202', 'Great Dane', '2754', '41'),
('G02000010', 'Great Dane', '2695', '41'),
('30106342', 'Great Dane', '2716', '41'),
('30106364', 'Great Dane', '2716', '41'),
('30100467', 'Great Dane', '2695', '41'),
('30100468', 'Great Dane', '2695', '41'),
('170003102', 'Hercules', '2919', '1'),
('10031775', 'Hyundai', '2695', '41'),
('H10031775', 'Hyundai', '2695', '41'),
('10031775 (77)', 'Hyundai', '2695', '41'),
('10031857', 'Hyundai', '2695', '41'),
('10059563B', 'Hyundai', '2695', '41'),
('10059563B (77)', 'Hyundai', '2695', '41'),
('10054157', 'Hyundai', '2695', '41'),
('10059703', 'Hyundai', '2695', '41'),
('1005970344', 'Hyundai', '2695', '41'),
('H10059703', 'Hyundai', '2695', '41'),
('10065864', 'Hyundai', '2695', '41'),
('10064857', 'Hyundai', '2695', '41'),
('10032827', 'Hyundai', '2718', '41'),
('10060082', 'Hyundai', '2718', '41'),
('10055269', 'Hyundai', '2718', '41'),
('30101511R', 'Great Dane', '2754', '41'),
('K9220991W', 'Kidron', '2695', '41'),
('K9220992W', 'Kidron', '2695', '41'),
('K9220993W', 'Kidron', '2695', '41'),
('K9220994W', 'Kidron', '2695', '41'),
('K9220997W', 'Kidron', '2695', '41'),
('K9220998W', 'Kidron', '2695', '41'),
('K9220998L', 'Kidron', '2695', '41'),
('K94920501', 'Kidron', '2695', '41'),
('K94920502', 'Kidron', '2695', '41'),
('K9492052B', 'Kidron', '2695', '41'),
('K94920507', 'Kidron', '2695', '41'),
('K9471901', 'Kidron', '2718', '1'),
('K9471902', 'Kidron', '2718', '1'),
('K951623-2', 'Kidron', '2718', '1'),
('K951623-3', 'Kidron', '2718', '1'),
('K9471903', 'Kidron', '2718', '1'),
('K9531703', 'Kidron', '2718', '1'),
('K9471903R', 'Kidron', '2718', '1'),
('K94719034', 'Kidron', '2718', '1'),
('K9471907', 'Kidron', '2718', '1'),
('K950079', 'Kidron', '2695', '41'),
('K950079-1', 'Kidron', '2695', '41'),
('K950698', 'Kidron', '2695', '41'),
('K950770', 'Kidron', '2695', '41'),
('K950770-1', 'Kidron', '2695', '41'),
('K950854', 'Kidron', '2695', '41'),
('K95104405', 'Kidron', '2695', '41'),
('K95104401', 'Kidron', '2695', '41'),
('K951044SP', 'Kidron', '2695', '41'),
('K95104402', 'Kidron', '2695', '41'),
('K95104406', 'Kidron', '2695', '41'),
('K95104407', 'Kidron', '2695', '41'),
('K95104403', 'Kidron', '2695', '41'),
('K95104404', 'Kidron', '2695', '41'),
('K95104408', 'Kidron', '2695', '41'),
('K95104409', 'Kidron', '2695', '41'),
('K953471-1', 'Kidron', '2695', '41'),
('K953471-2', 'Kidron', '2695', '41'),
('K95319307', 'Kidron', '2695', '41'),
('K95319305', 'Kidron', '2695', '41'),
('K95319306', 'Kidron', '2695', '41'),
('K95319308', 'Kidron', '2695', '41'),
('4X1015XCP', 'Lifeline Mobile', '2695', '41'),
('710035Q', 'Manac', '2695', '41'),
('02200062', 'Marion Body', '2919', '1'),
('02200063', 'Marion Body', '2919', '1'),
('02200067', 'Marion Body', '2919', '1'),
('02200068', 'Marion Body', '2919', '1'),
('2000600C', 'Marion Body', '2695', '41'),
('2919955', 'Mickey Body', '2919', '1'),
('2919955G', 'Mickey Body', '2919', '1'),
('6005400G', 'Mickey Body', '2919', '1'),
('29191015', 'Mickey Body', '2919', '1'),
('29191015G', 'Mickey Body', '2919', '1'),
('29191015D', 'Demountable', '2919', '1'),
('6005400', 'Mickey Body', '2919', '1'),
('6005401', 'Mickey Body', '2919', '1'),
('6005401G', 'Mickey Body', '2919', '1'),
('MEV-13908', 'Mickey Body', '2754', '41'),
('6604945', 'Mickey Body', '2754', '41'),
('010002H', 'Morgan', '2919', '1'),
('0100213', 'Morgan', '2919', '1'),
('010021H-B', 'Morgan', '2919', '1'),
('010021H', 'Morgan', '2919', '1'),
('010021HC', 'Morgan', '2919', '1'),
('010025HC', 'Morgan', '2919', '1'),
('010025H', 'Morgan', '2919', '1'),
('0100252', 'Morgan', '2919', '1'),
('0100252C', 'Morgan', '2919', '1'),
('010027C', 'Morgan', '2919', '1'),
('010025HB', 'Multivan', '2919', '1'),
('GB0075B', 'Multivan', '2919', '1'),
('GB0075W', 'Multivan', '2919', '1'),
('GB0075P', 'Multivan', '2919', '1'),
('GB0085B', 'Multivan', '2695', '41'),
('GB0085P', 'Multivan', '2695', '41'),
('GB0085W', 'Multivan', '2695', '41'),
('GB0102P', 'Multivan', '2919', '1'),
('GB0102W', 'Multivan', '2919', '1'),
('GB0102B', 'Multivan', '2919', '1'),
('GB0130', 'Multivan', '2919', '1'),
('GB0138', 'Multivan', '2919', '1'),
('GB010201P', 'Multivan', '2919', '1'),
('A44676-95', 'Dorsey', '2695', '41'),
('A44676101', 'Dorsey', '2695', '41'),
('A44927-93', 'Dorsey', '2695', '41'),
('A44927-99', 'Dorsey', '2695', '41'),
('17PRTAG18', 'Dorsey', '2695', '41'),
('A4492799B', 'Pitts', '2695', '41'),
('04114001', 'Stoughton', '2806', '41'),
('184540070', 'Stoughton', '2982', '41'),
('10113269S', 'Stoughton', '2695', '41'),
('184540010', 'Stoughton', '2982', '41'),
('184540010B', 'Stoughton', '2982', '41'),
('292650017', 'Stoughton', '2982', '41'),
('41512', 'Strick', '2990', '41'),
('120001009', 'Supreme', '2919', '1'),
('US010021H', 'U.S. Truck', '2919', '1'),
('US09562G', 'U.S. Truck', '2919', '1'),
('US010025G', 'U.S. Truck', '2919', '1'),
('US010025H', 'U.S. Truck', '2919', '1'),
('UST2360C', 'U.S. Truck', '2718', '1'),
('UST2361C', 'U.S. Truck', '2718', '1'),
('22500006', 'Utilimaster', '2718', '1'),
('22500007', 'Utilimaster', '2718', '1'),
('22822491', 'Utilimaster', '2919', '1'),
('22822490', 'Utilimaster', '2919', '1'),
('4257599', 'Utility', '2695', '41'),
('42575109', 'Utility', '2695', '41'),
('4257599HV', 'Utility', '2559', '41'),
('42575117', 'Utility', '2744', '41'),
('42575118', 'Utility', '2695', '41'),
('42575119', 'Utility', '2695', '41'),
('42575120', 'Utility', '2744', '41'),
('42575124', 'Utility', '2716', '41'),
('42575127', 'Utility', '2716', '41'),
('42575100W', 'Utility', '2718', '41'),
('425759575', 'Utility', '2718', '41'),
('4257599X', 'Utility', '2695', '41'),
('42575117X', 'Utility', '2744', '41'),
('42575118X', 'Utility', '2695', '41'),
('42575119X', 'Utility', '2695', '41'),
('42575120X', 'Utility', '2744', '41'),
('XMB00028', 'Vanguard', '2695', '41'),
('XMB00009', 'Vanguard', '2695', '41'),
('XMB00028B', 'Vanguard', '2695', '41'),
('XMB00009G', 'Vanguard', '2695', '41'),
('XMB00009P', 'Vanguard', '2695', '41'),
('XMB0009PA', 'Aurora', '2695', '41'),
('XMB00011', 'Vanguard', '2754', '41'),
('XMB00030', 'Vanguard', '2754', '41'),
('XMB00029', 'Vanguard', '2718', '41'),
('XMB00031', 'Vanguard', '2718', '41'),
('XMB00017', 'Vanguard', '2718', '41'),
('02000010V', 'Wabash', '2695', '41'),
('02000141V', 'Wabash', '2695', '41'),
('02000299', 'Wabash', '2695', '41'),
('02001100', 'Wabash', '2695', '41'),
('02000393', 'Wabash', '2695', '41'),
('02000403', 'Wabash', '2695', '41'),
('02000187V', 'Wabash', '2695', '41'),
('02000772', 'Wabash', '2695', '41'),
('02000840', 'Wabash', '2695', '41'),
('02000878A', 'Wabash', '2990', '41'),
('02000883', 'Wabash', '2990', '41'),
('02000884', 'Wabash', '2990', '41'),
('02009032', 'Wabash', '2990', '41'),
('02000994V', 'Wabash', '2695', '41'),
('02001050', 'Wabash', '2695', '41'),
('02001051V', 'Wabash', '2695', '41'),
('02001052V', 'Wabash', '2695', '41'),
('02001077A', 'Wabash', '2990', '41'),
('02001081', 'Wabash', '2990', '41'),
('02001121A', 'Wabash', '2990', '41'),
('02001141V', 'Wabash', '2559', '41'),
('02001142V', 'Wabash', '2559', '41'),
('02001145V', 'Wabash', '2695', '41'),
('02001146V', 'Wabash', '2695', '41'),
('02001472', 'Wabash', '2695', '41'),
('02001473', 'Wabash', '2695', '41'),
('02001475A', 'Wabash', '2990', '41'),
('02001480', 'Wabash', '2695', '41'),
('02002141V', 'Wabash', '2559', '41'),
('02002141H', 'Wabash', '2559', '41'),
('02002706', 'Wabash', '2695', '41'),
('02002707', 'Wabash', '2695', '41'),
('02002708', 'Wabash', '2990', '41'),
('02002709', 'Wabash', '2990', '41'),
('02004899', 'Wabash', '2695', '41'),
('02002710', 'Wabash', '2695', '41'),
('02005504', 'Wabash', '2695', '41'),
('02002711', 'Wabash', '2695', '41'),
('02003253', 'Wabash', '2695', '41'),
('02003255', 'Wabash', '2695', '41'),
('02003258', 'Wabash', '2990', '41'),
('02002712', 'Wabash', '2990', '41'),
('02005081', 'Wabash', '2990', '41'),
('02004441', 'Wabash', '2695', '41'),
('02000643', 'Wabash', '2695', '41'),
('02004892', 'Wabash', '2695', '41'),
('02003741', 'Wabash', '2695', '41'),
('02005282', 'Wabash', '2695', '41'),
('02003263', 'Wabash', '2695', '41'),
('02003257', 'Wabash', '2695', '41'),
('02004893', 'Wabash', '2695', '41'),
('02004897', 'Wabash', '2990', '41'),
('02005798', 'Wabash', '2695', '41'),
('02005814', 'Wabash', '2695', '41'),
('02005815', 'Wabash', '2695', '41'),
('02005383', 'Wabash', '2695', '41'),
('02005639', 'Wabash', '2559', '41'),
('02006777', 'Wabash', '2718', '41'),
('02006700', 'Wabash', '2718', '41'),
('02006625', 'Wabash', '2718', '41'),
('02006661', 'Wabash', '2718', '41'),
('02006664', 'Wabash', '2718', '41'),
('02006681', 'Wabash', '2718', '41'),
('02007673', 'Wabash', '2695', '41'),
('02007320', 'Wabash', '2695', '41'),
('02009016', 'Wabash', '2695', '41'),
('02009018', 'Wabash', '2695', '41'),
('02005089', 'Wabash', '2559', '41'),
('53004039', 'Wabash', '8695', '41'),
('TST03850', 'Wabash', '2695', '41'),
('TST03851', 'Wabash', '2695', '41'),
('TST03852', 'Wabash', '2695', '41'),
('TST03855', 'Wabash', '2695', '41'),
('02001211A', 'Wabash', '2990', '41'),
('3307681', 'XPO', '2695', '41'),
('01251004 (P)', 'Wabash', '2919', '1'),
('01251003 (P)', 'Wabash', '2919', '1'),
('01251005 (P)', 'Wabash', '2919', '1'),
('01251022 (P)', 'Wabash', '2695', '1'),
('010012 (P)', 'Morgan', '2919', '1'),
('0100121 (P)', 'Morgan', '2919', '1'),
('0100121C (P)', 'Morgan', '2919', '1'),
('40102687 (P)', 'Great Dane', '2695', '41'),
('40103283 (P)', 'Great Dane', '2716', '41'),
('10150P (P)', 'Aurora', '2695', '41'),
('012050034 (P)', 'Wabash', '2695', '41'),
('012050 (P)', 'Wabash', '2695', '1'),
('012009 (P)', 'Wabash', '2695', '1'),
('012057 (P)', 'Wabash', '2695', '1'),
('GB0151P (P)', 'Multivan', '2919', '1'),
('02200074 (P)', 'Marion Body', '2919', '1'),
('60901701X (P)', 'Great Dane', '2720', '41'),
('26951015P (P)', 'Durabody', '2695', '41'),
('26959638P (P)', 'Demountable Con', '2695', '1'),
('26951011P (P)', 'Demountable Con', '2695', '41'),
('26951013P (P)', 'Demountable Con', '2695', '41'),
('271810188 (P)', 'Utility', '2718', '41'),
('2695102P (P)', 'Aurora', '2695', '41'),
('2806102P (P)', 'Aurora', '2806', '41'),
('XMB4X102P (P)', 'Vanguard', '2695', '41'),
('2718102P (P)', 'Aurora', '2718', '41'),
('2919102P (P)', 'Fleet', '2919', '1'),
('700040071 (P)', 'Brown Cargo', '2695', '41'),
('26959131 (P)', 'Brown Cargo', '2695', '41'),
('26959562P (P)', 'Brown Cargo', '2695', '41'),
('269510188 (P)', 'Aurora', '2695', '41'),
('X0200952 (P)', 'Wabash', '2919', '1'),
('X0200954 (P)', 'Wabash', '2919', '1'),
('012101054 (P)', 'Wabash', '2919', '1'),
('010030 (P)', 'J Van Body', '2919', '1'),
('XMB00009P (P)', 'Vanguard', '2695', '41'),
('010002H (P)', 'Morgan', '2919', '1'),
('6005400 (P)', 'Mickey', '2919', '1'),
('6005401 (P)', 'Mickey', '2919', '1'),
('010025H (P)', 'Morgan', '2919', '1'),
('02006700 (P)', 'Wabash', '2718', '41'),
('33166 (P)', 'Great Dane', '2720', '41'),
('ACF51177C (P)', 'Aurora', '2695', '41'),
('A14089PNT (P)', 'Aurora', '2695', '41'),
('X002101C (P)', 'Wabash', '2718', '1'),
('26951015P', 'Durabody', '2695', '41'),
('40102385G (PU)', 'Great Dane', '2695', '41'),
('40102385B (PU)', 'Marshall', '2695', '41'),
('40102386B (PU)', 'Marshall', '2720', '41'),
('40102386G (PU)', 'Great Dane', '2720', '41'),
('40103246G (PU)', 'Great Dane', '2754', '41'),
('40102391X (PU)', 'Great Dane', '2695', '41'),
('40102391Y (PU)', 'Great Dane', '2695', '41'),
('30100466 (PU)', 'Great Dane', '2695', '41'),
('40102386M (PU)', 'Great Dane', '2720', '41'),
('40102385X (PU)', 'Great Dane', '2695', '41'),
('40103246S (PU)', 'Great Dane', '2754', '41'),
('40102385Y (PU)', 'Marshall', '2695', '41'),
('40102385S (PU)', 'Great Dane', '2695', '41'),
('40102385R (PU)', 'Great Dane', '2695', '41'),
('40102386S (PU)', 'Great Dane', '2720', '41'),
('40102386Z (PU)', 'Great Dane', '2720', '41'),
('40102386Y (PU)', 'Great Dane', '2720', '41'),
('30101515 (PU)', 'Great Dane', '2695', '41'),
('30101516 (PU)', 'Great Dane', '2695', '41'),
('30101514 (PU)', 'Great Dane', '2720', '41'),
('30101519 (PU)', 'Great Dane', '2720', '41'),
('30101520 (PU)', 'Great Dane', '2716', '41'),
('30101517 (PU)', 'Great Dane', '2716', '41'),
('30106293G (PU)', 'Great Dane', '2695', '41'),
('27169503H (PU)', 'Marshall', '2716', '41'),
('27161013H (PU)', 'Marshall', '2716', '41'),
('40103242B (PU)', 'Great Dane', '2716', '41'),
('40105344B (PU)', 'Great Dane', '2695', '41'),
('14293B (PU)', 'Great Dane', '2559', '41'),
('40103247X (PU)', 'Great Dane', '2754', '41'),
('40103247B (PU)', 'Great Dane', '2754', '41'),
('2695935H (PU)', 'Evolve Solutions', '2695', '41'),
('A44927-93 (PU)', 'Dorsey', '2695', '41'),
('A44927-99 (PU)', 'Dorsey', '2695', '41'),
('K95104401 (PU)', 'Kidron', '2695', '41'),
('K95104402 (PU)', 'Kidron', '2695', '41'),
('K95319306 (PU)', 'Kidron', '2695', '41'),
('K95104403 (PU)', 'Kidron', '2695', '41'),
('K95104404 (PU)', 'Kidron', '2695', '41'),
('42575117 (PU)', 'Utility', '2744', '41'),
('3257544B (PU)', 'Utility', '2744', '41'),
('42575118 (PU)', 'Utility', '2695', '41'),
('42575120 (PU)', 'Utility', '2744', '41'),
('3257538B (PU)', 'Marshall', '2744', '41'),
('3257538DB (PU)', 'Marshall', '2695', '41'),
('3257540B (PU)', 'Marshall', '2744', '41'),
('40105853 (PU)', 'Marshall', '2559', '41'),
('02002710 (PU)', 'Wabash', '2695', '41'),
('02002711 (PU)', 'Wabash', '2695', '41'),
('02300643 (PU)', 'Wabash', '2695', '41'),
('02300620 (PU)', 'Wabash', '2695', '41'),
('02005282 (PU)', 'Wabash', '2695', '41'),
('02001100 (PU)', 'Wabash', '2695', '41'),
('02007320 (PU)', 'Wabash', '2695', '41'),
('02300446 (PU)', 'Wabash', '2695', '41'),
('02003741 (PU)', 'Wabash', '2695', '41'),
('02002712 (PU)', 'Wabash', '2990', '41'),
('02001050 (PU)', 'Wabash', '2695', '41'),
('02001081 (PU)', 'Wabash', '2990', '41'),
('53003905 (PU)', 'Wabash', '2990', '41'),
('02000643 (PU)', 'Wabash', '2695', '41'),
('02007673 (PU)', 'Wabash', '2695', '41'),
('TST03851 (PU)', 'Wabash', '2695', '41'),
('6005400 (PU)', 'Mickey', '2919', '1'),
('6005401 (PU)', 'Mickey', '2919', '1'),
('MEV-13908 (PU)', 'Mickey Body', '2754', '41');

MERGE FAB.Products AS target
USING (
    SELECT s.PartNumber, c.CustomerId, sec.SectionId, g.GradeId
    FROM #ProductStaging s
    LEFT JOIN FAB.catCustomers c ON c.CustomerName = s.CustomerName
    LEFT JOIN FAB.catSections sec ON sec.SectionName = s.SectionName
    LEFT JOIN FAB.catGrades g ON g.GradeName = s.GradeName
) AS source
ON target.PartNumber = source.PartNumber
WHEN NOT MATCHED THEN 
    INSERT (PartNumber, CustomerId, SectionId, GradeId, CreatedBy) 
    VALUES (source.PartNumber, source.CustomerId, source.SectionId, source.GradeId, 'seed')
WHEN MATCHED THEN
    UPDATE SET CustomerId = source.CustomerId, SectionId = source.SectionId, 
               GradeId = source.GradeId, ModifiedBy = 'seed', ModifiedDate = GETUTCDATE();

PRINT CONCAT('Products: ', @@ROWCOUNT, ' merged');
DROP TABLE #ProductStaging;

-- ============================================================
-- 4. WELDSPECS (using OUTER APPLY for ClipId lookup)
-- ============================================================
PRINT '';
PRINT '=== Step 4: WeldSpecs ===';

IF OBJECT_ID('tempdb..#WeldSpecStaging') IS NOT NULL DROP TABLE #WeldSpecStaging;

CREATE TABLE #WeldSpecStaging (
    PartNumber VARCHAR(50),
    CutLength DECIMAL(10,4),
    FinishedLength DECIMAL(10,4),
    Tolerance VARCHAR(20),
    Program VARCHAR(20),
    CleaningCycle INT,
    WebHoles VARCHAR(15),
    EntryKeeper VARCHAR(20),
    ExitKeeper VARCHAR(20),
    BottomKeeper VARCHAR(20),
    NextOp VARCHAR(20),
    BundleCol INT,
    BundleRow INT,
    BundleCount INT,
    StickSize VARCHAR(50),
    GroovedTimber VARCHAR(20),
    BandLength VARCHAR(20),
    Comments VARCHAR(500),
    ClipWidth DECIMAL(6,3),
    ClipHeight DECIMAL(6,3),
    ClipHolePattern VARCHAR(30),
    ClipHoleGage VARCHAR(20),
    ClipSideGage VARCHAR(20)
);

INSERT INTO #WeldSpecStaging VALUES
('3526141C', 94.732, 95.0, '+.010/-.100', '40', 8, 'No', 'A2', 'A2', 'X', 'Paint', 6, 16, 96, 'No Sticks', '38"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('35261410C', 100.732, 101.0, '+.010/-.100', '40', 8, 'No', 'A2', 'A2', 'X', 'Paint', 6, 16, 96, 'No Sticks', '38"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('31098-01', 91.21875, 91.5, '+.010/-.100', '30', 8, 'No', 'A3', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x35"', '33"', NULL, NULL, 2.875, 3.25, '2 Hole-0.313', 'W26 / W26L', 'W25'),
('31098', 95.11, 95.38, '+.010/-.100', '30', 8, 'No', 'A3', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x35"', '33"', NULL, NULL, 2.875, 3.25, '2 Hole-0.313', 'W26 / W26L', 'W25'),
('31099', 101.11, 101.38, '+.010/-.100', '30', 8, 'No', 'A3', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x35"', '33"', NULL, NULL, 2.875, 3.25, '2 Hole-0.313', 'W26 / W26L', 'W25'),
('A33166', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'A0', 'A0', 'X', 'Band to Ship', 7, 15, 105, '1/2"x1"x43"', '41"', NULL, NULL, 3.5, 3.5, 'Blank', 'W18', 'W16'),
('A02000010', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.406', 'W5', 'W1'),
('A02000141', 100.11, 100.38, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.406', 'W5', 'W1'),
('A0910076', 100.49, 100.76, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Paint', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.938, '4 Hole-0.406', 'W44', 'W4'),
('A02000393', 101.36, 101.63, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('A02001133', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.406', 'W5', 'W1'),
('ACF51177C', 101.112, 101.38, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 16, 96, '1/2"x1"x39"', '38"', NULL, NULL, 4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('A10031775', 101.23, 101.5, '+.040/-.070', '47', 8, 'No', 'B1', 'B1', 'X', 'Wax', 11, 8, 88, '1/2"x1"x43"', '38"', NULL, NULL, 5.197, 4.016, '4 Hole-0.406', 'Flush to Bottom', 'W48'),
('A18454001', 100.85, 101.12, '+.010/-.100', '44', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.938, '4 Hole-0.406', 'W44', 'W4'),
('A3-101BC', 101.3, 101.62, '+.010/-.100', '40', 8, 'No', 'A2', 'A2', 'X', 'Wax', 5, 18, 90, '1/2"x1"x39"', '38"', NULL, NULL, 4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('20071290', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 11, 8, 88, '1/2"x1"x43"', '38"', NULL, NULL, 5.197, 4.016, '5 Hole-0.406', 'Flush to Bottom', 'W48'),
('BCV010002', 89.35, 89.62, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, 'No Sticks', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('BCV010021', 95.35, 95.62, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, 'No Sticks', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('BCV010025', 101.35, 101.62, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, 'No Sticks', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('BCV10025G', 101.35, 101.62, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('291995325', 95.06, 95.33, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, 'No Sticks', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('291910132', 101.06, 101.33, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, 'No Sticks', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('26959131P', 91.31, 91.31, '+.04/-.04', NULL, NULL, 'No', NULL, NULL, NULL, 'Paint', NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('26959562P', 95.62, 95.62, '+.040/-.040', NULL, NULL, 'No', NULL, NULL, NULL, 'Paint', NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('26951015D', 101.23, 101.5, '+040/-.070', '40', 8, 'No', 'A2', 'A2', 'X', 'Wax', 5, 18, 90, '1/2"x1"x43"', '41"', NULL, '5H x 18W/Hyundai Stock', 4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('FCE7662-7', 101.112, 101.38, '+.010/-.100', '40', 6, 'No', 'A1', 'A1', 'X', 'Paint', 6, 16, 96, 'No Sticks', '38"', NULL, NULL, 4.0, 2.5, '4 Hole-0.406', 'W37', 'W4'),
('FCF88914C', 100.852, 101.12, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 16, 96, 'No Sticks', '38"', NULL, NULL, 4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('FCF51177C', 101.112, 101.38, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 16, 96, 'No Sticks', '38"', NULL, NULL, 4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('SP0019001', 101.112, 101.38, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Band to Ship', 5, 18, 90, '1/2"x1"x43"', '38"', NULL, NULL, 3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
('SP0019002', 101.112, 101.38, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Wax', 5, 18, 90, '1/2"x1"x43"', '38"', NULL, NULL, 3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
('F03510100', 101.11, 101.38, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Paint', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101506', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.5, 'Blank', 'W18', 'W1'),
('30101511', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101536', 94.37, 94.64, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101511B', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101511Z', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101511M', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 4.0, 5.75, 'Blank', 'Flush to Bottom', 'W47'),
('30101511G', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101514', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101514B', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1/2"x1"x43"', '39"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101514G', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101515', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101515B', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101515G', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101516', 95.03, 95.3, '+.010/-.100', '53/41', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101516B', 95.03, 95.3, '+.010/-.100', '53', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101516G', 95.03, 95.3, '+.010/-.100', '53', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101517', 101.39, 101.66, '+.010/-.100', '33', 6, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19', 'W17'),
('30101517B', 101.39, 101.66, '+.010/-.100', '33', 6, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19', 'W17'),
('30101517G', 101.39, 101.66, '+.010/-.100', '33', 6, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19', 'W17'),
('30101518', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101518B', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101518G', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101518M', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.75, 4.0, 'Blank', 'Flush to Bottom', 'W47'),
('30101518T', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30100041', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4-Drain Holes only', 'Flush to Bottom', 'W1'),
('30100228', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30100178', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4-Drain Holes only', 'Flush to Bottom', 'W1'),
('30101519', 95.03, 95.3, '+.010/-.100', '53/41', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101519B', 95.03, 95.3, '+.010/-.100', '53', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101519G', 95.03, 95.3, '+.010/-.100', '53', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30101520', 95.03, 95.3, '+.010/-.100', '33', 6, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19', 'W17'),
('30101520B', 95.03, 95.3, '+.010/-.100', '33', 6, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19', 'W17'),
('30101520G', 95.03, 95.3, '+.010/-.100', '33', 6, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19', 'W17'),
('30101523', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19', 'W1'),
('30101524', 100.73, 101.0, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W17'),
('30101524G', 100.73, 101.0, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W17'),
('30101524M', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.75, 4.0, 'Blank', 'Flush to Bottom', 'W47'),
('30101526', 95.03, 95.3, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101526B', 95.03, 95.3, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30101527', 95.03, 95.3, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19', 'W17'),
('30106314B', 95.03, 95.3, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W17'),
('30101534', 101.39, 101.66, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W17'),
('30101534B', 101.39, 101.66, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W17'),
('30101534G', 101.39, 101.66, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 4.0, '4-Drain Holes only', 'Flush to Bottom', 'W17'),
('30101539', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19', 'W1'),
('30106221', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30106278', 101.39, 101.66, '+.010/-.100', '30', 8, 'No', 'A2', 'A2', 'X', 'Wax', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.5, 'Blank', 'Flush to Bottom', 'W46'),
('30106278B', 101.39, 101.66, '+.010/-.100', '30', 8, 'No', 'A2', 'A2', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.5, 'Blank', 'Flush to Bottom', 'W46'),
('30106293G', 100.73, 101.0, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30106262', 100.73, 101.0, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.85, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30106296', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '8 Hole-.0406', 'Flush to Bottom', 'W1'),
('30106252', 100.73, 101.0, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Wax', 8, 11, 88, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W17'),
('30106298', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '8 Hole-.0406', 'Flush to Bottom', 'W1'),
('30106326', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Drain Holes', 'Flush to Bottom', 'W1'),
('30106336', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.75, 4.0, 'Blank', 'Flush to Bottom', 'W47'),
('30106346', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('30106358', 95.03, 95.3, '+.010/-.100', '51', 8, 'No', 'A2', 'A2', 'X', 'Wax', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.5, 'Blank', 'Flush to Bottom', 'W46'),
('30106358B', 95.03, 95.3, '+.010/-.100', '51', 8, 'No', 'A2', 'A2', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.5, 'Blank', 'Flush to Bottom', 'W46'),
('30110002', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('30110002D', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('30100042', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('30100025', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('30100024', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x44"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('30100024B', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x44"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('30100043', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30100044', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '8 Hole-.0406', 'Flush to Bottom', 'W1'),
('30100149', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '8 Hole-.0406', 'Flush to Bottom', 'W1'),
('30110041', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('30110041D', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('30110096', 100.73, 101.0, '+.010/-.100', NULL, NULL, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', NULL),
('30110130', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Drain Holes', 'Flush to Bottom', 'W1'),
('30110132', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '8 Hole-.0406', 'Flush to Bottom', 'W1'),
('30110134', 100.73, 101.0, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 4.0, '8 hole-.0406 3" Pattern', 'Flush to Bottom', 'W17'),
('30110138', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '8 Hole-.0406', 'Flush to Bottom', 'W1'),
('30110106', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('30133446', 101.39, 101.66, '+.010/-.100', NULL, NULL, 'Yes', 'A0', 'A0', 'X', 'Wax', 8, 15, 120, '1/2"x1"x35"', '33"', NULL, NULL, 3.5, 3.0, 'Blank', 'W18', 'W16'),
('33166', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'A0', 'A0', 'X', 'Paint', 7, 15, 105, 'No Sticks', '35"', NULL, NULL, 3.5, 3.5, 'Blank', 'W18', 'W16'),
('33016', 95.03, 95.3, '+.010/-.100', '40', 8, 'Yes', 'A0', 'A0', 'X', 'Paint', 7, 15, 105, '1/2"x1"x39"', '35"', NULL, NULL, 3.5, 3.5, 'Blank', 'W18', 'W16'),
('40103202', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('G02000010', 100.73, 101.0, '+.010/-.100', NULL, NULL, 'No', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.406', NULL, NULL),
('30106342', 101.39, 101.66, '+.010/-.100', '33', 6, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 4.0, 'Blank', 'W19', 'W17'),
('30106364', 101.39, 101.66, '+.010/-.100', '33', 6, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 11, 88, '1"x1"x43"', '38"', NULL, NULL, 5.0, 4.0, 'Drain Hole', 'Flush to Bottom', 'W17'),
('30100467', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'W19 Flush to Bottom', 'W1'),
('30100468', 101.39, 101.66, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'W19 Flush to Bottom', 'W1'),
('170003102', 100.73, 101.0, '+.010/-.100', '30', 8, 'No', 'A3', NULL, NULL, 'Band to Ship', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 4.25, 2.75, '4 Hole-0.406', NULL, NULL),
('10031775', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Wax', 11, 8, 88, '1/2"x1"x43"', '41"', NULL, NULL, 5.197, 4.016, '5 Hole-0.406', 'Flush to Bottom', 'W48'),
('H10031775', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Wax', 11, 8, 88, '1/2"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '5 Hole-0.406', 'Flush to Bottom', 'W48'),
('10031775 (77)', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Wax', 11, 7, 77, '1/2"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '5 Hole-0.406', 'Flush to Bottom', 'W48'),
('10031857', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 11, 8, 88, '1/2"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '5 Hole-0.406', 'Flush to Bottom', 'W48'),
('10059563B', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 11, 8, 88, '1/2"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '1-Hole', 'Flush to Bottom', 'W48'),
('10059563B (77)', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 11, 7, 77, '1/2"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '1-Hole', 'Flush to Bottom', 'W48'),
('10054157', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 11, 8, 88, '1/2"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, 'Blank', 'Flush to Bottom', 'W48'),
('10059703', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Wax', 11, 8, 88, '1/2"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '1-Hole', 'Flush to Bottom', 'W48'),
('1005970344', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Wax', 11, 8, 88, '1/2"x1"x43"', '39"Heat Treated', NULL, '2 @ 44 = 88Pc. Bundle', 5.197, 4.016, '1-Hole', 'Flush to Bottom', 'W48'),
('H10059703', 101.23, 101.5, '+.015/-.100', '47', 8, 'No', 'B1', 'B1', 'X', 'Wax', 11, 8, 88, '1/2"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '1-Hole', 'Flush to Bottom', 'W48'),
('10065864', 101.23, 101.5, '+.015/-.100', '37', 8, 'No', 'B1', 'B1', 'X', 'Wax', 15, 6, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.197, 4.016, '4 Hole-0.406', 'Flush to Bottom', 'W48'),
('10064857', 101.23, 101.5, '+.015/-.100', '34', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 15, 6, 90, '1/2"x1"x43"', '39"Heat Treated', NULL, NULL, 3.197, 4.016, '4 Hole-0.406', 'Flush to Bottom', NULL),
('10032827', 101.23, 101.5, '+.015/-.100', '37', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 11, 8, 88, '1"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '5 Hole-0.406', 'Flush to Bottom', 'W48'),
('10060082', 101.23, 101.5, '+.015/-.100', '37', 8, 'No', 'B1', 'B1', 'X', 'Wax', 11, 8, 88, '1"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '1-Hole', 'Flush to Bottom', 'W48'),
('10055269', 94.93, 95.2, '+.015/-.100', NULL, 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 11, 8, 88, '1"x1"x43"', '39"Heat Treated', NULL, NULL, 5.197, 4.016, '5 Hole-0.406', 'Flush to Bottom', 'W48'),
('30101511R', 101.39, 101.66, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, 'Blank', 'Flush to Bottom', 'W1'),
('K9220991W', 95.17, 95.44, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K9220992W', 101.17, 101.44, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K9220993W', 94.92, 95.19, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K9220994W', 94.67, 94.94, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K9220997W', 100.92, 101.19, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K9220998W', 100.17, 100.44, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K9220998L', 100.67, 100.94, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K94920501', 95.17, 95.44, '+.010/-.100', '40', 8, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K94920502', 101.17, 101.44, '+.010/-.100', '40', 8, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K9492052B', 101.17, 101.44, '+.010/-.100', '40', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K94920507', 100.92, 101.19, '+.010/-.100', '40', 8, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 3.75, 2.25, '4 Hole-0.406', 'W34', 'W11'),
('K9471901', 95.26, 95.53, '+040/-.070', '30', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.25, '4 Hole-0.406', 'W31', 'W30'),
('K9471902', 95.17, 95.44, '+040/-.070', '51', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.25, '4 Hole-0.406', 'W31', 'W30'),
('K951623-2', 95.17, 95.44, '+040/-.070', '51', 8, 'No', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.25, '4 Hole-0.406', 'W31', 'W30'),
('K951623-3', 101.17, 101.44, '+040/-.070', '30', 8, 'No', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.25, '4 Hole-0.406', 'W31', 'W30'),
('K9471903', 101.17, 101.44, '+040/-.070', '30', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.25, '4 Hole-0.406', 'W31', 'W30'),
('K9531703', 101.17, 101.44, '+040/-.070', '30', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.25, '2 Hole-0.406', 'W31', 'W30'),
('K9471903R', 99.17, 99.44, '+040/-.070', '30', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, 'Turn 2hole clip to side', 3.0, 2.25, '4 Hole-0.406', 'W31', 'W30'),
('K94719034', 97.17, 97.44, '+040/-.070', '30', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.25, '4 Hole-0.406', 'W31', 'W30'),
('K9471907', 89.17, 89.44, '+040/-.070', '38', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 18, 108, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.25, '4 Hole-0.406', 'W31', 'W30'),
('K950079', 101.36, 101.63, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K950079-1', 101.36, 101.63, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K950698', 95.362, 95.63, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K950770', 101.11, 101.38, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K950770-1', 101.11, 101.38, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K950854', 95.11, 95.38, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95104405', 101.36, 101.63, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95104401', 101.36, 101.63, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K951044SP', 101.11, 101.38, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95104402', 101.11, 101.38, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95104406', 101.11, 101.38, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95104407', 95.36, 95.63, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95104403', 95.36, 95.63, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95104404', 95.11, 95.38, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95104408', 95.11, 95.38, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95104409', 100.59, 100.86, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K953471-1', 95.17, 95.44, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K953471-2', 101.17, 101.44, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W42', 'W4'),
('K95319307', 95.36, 95.63, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x39"', '41"', NULL, 'Rotate Clips', 3.0, 4.0, '4 Hole-0.406', 'W42', NULL),
('K95319305', 101.36, 101.63, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x39"', '35"', '13''', 'Rotate Clips', 3.0, 4.0, '4 Hole-0.406', 'W42', NULL),
('K95319306', 101.11, 101.38, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x39"', '35"', '13''', 'Rotate Clips', 3.0, 4.0, '4 Hole-0.406', 'W42', NULL),
('K95319308', 95.11, 95.38, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x39"', '41"', NULL, 'Rotate Clips', 3.0, 4.0, '4 Hole-0.406', 'W42', NULL),
('4X1015XCP', 101.23, 101.5, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Paint', 6, 16, 96, 'No Sticks', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('710035Q', 101.502, 101.77, '+.010/-.100', '48', 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.5, 'Blank', 'Flush to Bottom', 'W1'),
('02200062', 95.35, 95.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, 'No Sticks', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('02200063', 101.35, 101.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, 'No Sticks', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('02200067', 95.35, 95.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('02200068', 101.35, 101.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('2000600C', 101.112, 101.38, '+.010/-.100', '40', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 16, 96, 'No Sticks', '38"', NULL, NULL, 4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('2919955', 95.23, 95.5, '+.010/-.100', '30', 8, 'No', 'A3', 'C1 =2.75"', 'X', 'Paint', 7, 14, 98, '1/2"x1"x43"', '41"', NULL, NULL, 4.25, 2.75, 'Blank', NULL, 'W22'),
('2919955G', 95.23, 95.5, '+.010/-.100', '30', 8, 'No', 'A3', 'C1 =2.75"', 'X', 'Band to Ship', 7, 14, 98, '1/2"x1"x43"', '41"', NULL, NULL, 4.25, 2.75, 'Blank', NULL, 'W22'),
('6005400G', 95.23, 95.5, '+.010/-.100', '30', 8, 'Yes', 'A3', 'C1 =2.75"', 'X', 'Band to Ship', 7, 14, 98, '1/2"x1"x43"', '41"', NULL, NULL, 4.25, 2.75, 'Blank', NULL, 'W22'),
('29191015', 101.23, 101.5, '+.010/-.100', '30', 8, 'No', 'A3', 'C1 =2.75"', 'X', 'Paint', 7, 14, 98, '1/2"x1"x43"', '41"', NULL, NULL, 4.25, 2.75, 'Blank', NULL, 'W22'),
('29191015G', 101.23, 101.5, '+.010/-.100', '30', 8, 'No', 'A3', 'C1 =2.75"', 'X', 'Band to Ship', 7, 14, 98, '1/2"x1"x43"', '41"', NULL, NULL, 4.25, 2.75, 'Blank', NULL, 'W22'),
('29191015D', 101.23, 101.5, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('6005400', 95.23, 95.5, '+.010/-.100', '30', 8, 'Yes', 'A3', 'C1 =2.75"', 'X', 'Paint', 7, 14, 98, '1/2"x1"x43"', '41"', NULL, NULL, 4.25, 2.75, 'Blank', NULL, 'W22'),
('6005401', 101.23, 101.5, '+.010/-.100', '30', 8, 'Yes', 'A3', 'C1 =2.75"', 'X', 'Paint', 7, 14, 98, '1/2"x1"x43"', '41"', NULL, NULL, 4.25, 2.75, 'Blank', '`', 'W22'),
('6005401G', 101.23, 101.5, '+.010/-.100', '30', 8, 'Yes', 'A3', 'C1 =2.75"', 'X', 'Band to Ship', 7, 14, 98, '1/2"x1"x43"', '41"', NULL, NULL, 4.25, 2.75, 'Blank', '`', 'W22'),
('MEV-13908', 101.36, 101.63, '+.010/-.100', '40', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x39"', '38"', NULL, NULL, 4.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'XX'),
('6604945', 95.35, 95.62, '+.010/-.100', '40', 8, 'Yes', 'A3', 'A3', 'X', 'Paint', 6, 16, 96, 'No Sticks', '38"', NULL, NULL, 4.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'XX'),
('010002H', 89.35, 89.62, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('0100213', 91.35, 91.62, '+.010/-.100', '37', 8, 'No', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('010021H-B', 95.35, 95.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('010021H', 95.35, 95.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('010021HC', 95.35, 95.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Paint', 5, 10, 50, '1/2"x1"x22"', '19"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('010025HC', 101.35, 101.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Paint', 5, 10, 50, '1/2"x1"x22"', '19"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('010025H', 101.35, 101.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('0100252', 101.35, 101.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('0100252C', 101.35, 101.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 5, 10, 50, '1/2"x1"x22"', '19"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('010027C', 71.35, 71.62, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, 'No Sticks', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('010025HB', 101.35, 101.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('GB0075B', 101.23, 101.5, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '35"', NULL, NULL, 3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
('GB0075W', 101.23, 101.5, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 18, 108, '1/2"x1"x39"', '35"', NULL, NULL, 3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
('GB0075P', 101.23, 101.5, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, '1/2"x1"x39"', '35"', NULL, NULL, 3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
('GB0085B', 101.23, 101.5, '+.010/-.100', '44', 8, 'No', 'A2', 'A2', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('GB0085P', 101.23, 101.5, '+.010/-.100', '44/39(160)', 8, 'No', 'A2', 'A2', 'X', 'Paint', 6, 16, 96, 'No Sticks', '35"', NULL, NULL, 4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('GB0085W', 101.23, 101.5, '+.010/-.100', '44', 8, 'No', 'A2', 'A2', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '35"', NULL, NULL, 4.0, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('GB0102P', 95.23, 95.5, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, '1/2"x1"x39"', '35"', NULL, NULL, 3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
('GB0102W', 95.23, 95.5, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Wax', 6, 18, 108, '1/2"x1"x39"', '35"', NULL, NULL, 3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
('GB0102B', 95.23, 95.5, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '35"', NULL, NULL, 3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
('GB0130', 91.23, 91.5, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '35"', NULL, NULL, 3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
('GB0138', 89.73, 90.0, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '35"', NULL, NULL, 3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
('GB010201P', 94.61, 94.88, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, 'No Sticks', '35"', NULL, NULL, 3.0, 2.0, '4 Hole-0.406', 'W14', 'W32'),
('A44676-95', 95.082, 95.35, '+040/-.070', '46', 6, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 5.125, 'Blank', NULL, 'W45'),
('A44676101', 101.082, 101.35, '+040/-.070', '46', 6, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 5.125, 'Blank', NULL, 'W45'),
('A44927-93', 93.572, 93.84, '+040/-.070', '54/46', 6, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 5.125, 'Blank', NULL, 'W45'),
('A44927-99', 99.572, 99.84, '+040/-.070', '54/46', 6, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 5.125, 'Blank', NULL, 'W45'),
('17PRTAG18', 99.572, 99.84, '+040/-.070', '46', 6, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 5.125, 'Blank', NULL, 'W45'),
('A4492799B', 99.57, 99.84, '+040/-.070', '46', 6, 'Yes', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 5.125, 'Blank', NULL, 'W45'),
('04114001', 100.67, 100.97, '+.010/-.100', NULL, NULL, 'No', NULL, NULL, 'X', NULL, 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 6.0, 2.125, 'Unknown', NULL, NULL),
('184540070', 100.45, 100.75, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.938, '4 Hole-0.406', 'W44', 'W4'),
('10113269S', 100.85, 101.12, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.938, '4 Hole-0.406', 'W44', 'W4'),
('184540010', 100.85, 101.12, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.938, '4 Hole-0.406', 'W44', 'W4'),
('184540010B', 100.85, 101.12, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.938, '4 Hole-0.406', 'W44', 'W4'),
('292650017', 100.85, 101.12, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.938, '4 Hole-0.406', 'W44', 'W4'),
('41512', 101.11, 101.38, '+.010/-.100', '30', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.406', 'Flush to Bottom', 'W1'),
('120001009', 101.11, 101.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '8 Hole-0.406', 'W7', 'W3'),
('US010021H', 95.35, 95.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, '1/2"x1"x43"', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('US09562G', 95.35, 95.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x43"', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('US010025G', 101.35, 101.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Band to Ship', 6, 18, 108, '1/2"x1"x39"', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('US010025H', 101.35, 101.62, '+.010/-.100', '30', 8, 'Yes', 'A3', 'A3', 'X', 'Paint', 6, 18, 108, '1/2"x1"x43"', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('UST2360C', 95.35, 95.62, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 96, '1/2"x1"x39"', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('UST2361C', 101.35, 101.62, '+.010/-.100', '30', 8, 'No', 'A3', 'A3', 'X', 'Paint', 6, 18, 96, '1/2"x1"x39"', '35"', NULL, NULL, 2.875, 2.0, '4 Hole-0.406', 'W33', 'W32'),
('22500006', 95.23, 95.5, '+.010/-.100', '30', 8, 'No', 'B1', 'A3', 'X', 'Wax', 8, 12, 96, '1/2"x1"x35"', '33"', NULL, NULL, 2.875, 3.25, '2 Hole-0.406', 'W36 / W36L', 'W35'),
('22500007', 101.23, 101.5, '+.010/-.100', '30', 8, 'No', 'B1', 'A3', 'X', 'Wax', 8, 12, 96, '1/2"x1"x35"', '33"', NULL, NULL, 2.875, 3.25, '2 Hole-0.406', 'W36 / W36L', 'W35'),
('22822491', 95.23, 95.5, '+.010/-.100', '30', 8, 'No', 'B1', 'A3', 'X', 'Wax', 8, 12, 96, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 3.25, '2 Hole-0.406', 'W36 / W36L', 'W38'),
('22822490', 101.23, 101.5, '+.010/-.100', '30', 8, 'No', 'B1', 'A3', 'X', 'Wax', 8, 12, 96, '1/2"x1"x39"', '38"', NULL, NULL, 2.875, 3.25, '2 Hole-0.406', 'W36 / W36L', 'W38'),
('4257599', 101.48, 101.75, '+.010/-.100', '41', 6, 'No', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
('42575109', 95.48, 95.75, '+.010/-.100', '41', 6, 'No', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
('4257599HV', 101.48, 101.75, '+.010/-.100', '41', 6, 'No', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
('42575117', 101.48, 101.75, '+.010/-.100', '41', 6, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
('42575118', 101.48, 101.75, '+.010/-.100', '41', 6, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
('42575119', 95.48, 95.75, '+.010/-.100', '41', 6, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
('42575120', 95.48, 95.75, '+.010/-.100', '54', 6, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '4 Hole-0.391', 'W12', 'W11'),
('42575124', 101.48, 101.75, '+.010/-.100', '30', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 15, 90, '1/2"x1"x44"', '41"', NULL, NULL, 3.0, 2.75, '4 Hole-0.391', 'W12', 'W13'),
('42575127', 95.48, 95.75, '+.010/-.100', '30', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 15, 90, '1/2"x1"x44"', '41"', NULL, NULL, 3.0, 2.75, '4 Hole-0.391', 'W12', 'W13'),
('42575100W', 101.48, 101.75, '+.010/-.100', '31', 6, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.907, '4 Hole-0.391', 'W40', 'W4'),
('425759575', 95.48, 95.75, '+.010/-.100', '31', 6, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 2.907, '4 Hole-0.391', 'W40', 'W4'),
('4257599X', 101.48, 101.75, '+.010/-.100', '41', 6, 'No', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '6 Hole-0.391', 'W12', 'W11'),
('42575117X', 101.48, 101.75, '+.010/-.100', '41', 6, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '6 Hole-0.391', 'W12', 'W11'),
('42575118X', 101.48, 101.75, '+.010/-.100', '41', 6, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '6 Hole-0.391', 'W12', 'W11'),
('42575119X', 95.48, 95.75, '+.010/-.100', '41', 6, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '6 Hole-0.391', 'W12', 'W11'),
('42575120X', 95.48, 95.75, '+.010/-.100', '41', 6, 'Yes', 'A3', 'A3', 'X', 'Wax', 6, 15, 90, '1/2"x1"x43"', '41"', NULL, NULL, 3.938, 2.25, '6 Hole-0.391', 'W12', 'W11'),
('XMB00028', 101.23, 101.5, '+.010/-.100', '42', 6, 'No', 'A2', 'A2', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.25, '4 Hole-0.406', 'W14', 'W49'),
('XMB00009', 101.23, 101.5, '+.010/-.100', '41', 6, 'No', 'A2', 'A2', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '38"', NULL, NULL, 4.25, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('XMB00028B', 101.23, 101.5, '+.010/-.100', '41', 6, 'No', 'A2', 'A2', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.25, '4 Hole-0.406', 'W14', 'W49'),
('XMB00009G', 101.23, 101.5, '+.010/-.100', '41', 6, 'No', 'A2', 'A2', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x39"', '38"', NULL, NULL, 4.25, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('XMB00009P', 101.23, 101.5, '+.010/-.100', '41', 6, 'No', 'A2', 'A2', 'X', 'Paint', 6, 16, 96, '1/2"x1"x39"', '38"', NULL, NULL, 4.25, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('XMB0009PA', 101.23, 101.5, '+.010/-.100', '41', 6, 'No', 'A2', 'A2', 'X', 'Paint', 6, 16, 96, '1/2"x1"x39"', '38"', NULL, NULL, 4.25, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('XMB00011', 101.23, 101.5, '+.010/-.100', '41', 6, 'No', 'A2', 'A2', 'X', 'Wax', 6, 16, 96, '1/2"x1"x39"', '38"', NULL, NULL, 4.25, 2.5, '4 Hole-0.406', 'W14', 'W15'),
('XMB00030', 101.23, 101.5, '+.010/-.100', '41', 6, 'No', 'A2', 'A2', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 4.25, '4 Hole-0.406', 'W14', 'W49'),
('XMB00029', 101.23, 101.5, '+.010/-.100', '31', 6, 'No', 'A2', 'A2', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 5.0, 3.25, '4 Hole-0.406', 'W41', 'W1'),
('XMB00031', 101.23, 101.5, '+.010/-.100', '31', 6, 'No', 'A2', 'A2', 'X', 'Wax', 6, 16, 96, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.25, '2 Hole-0.406', 'W14', 'W49'),
('XMB00017', 101.23, 101.5, '+.010/-.100', '31', 6, 'No', 'A1', 'A1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x39"', '38"', NULL, NULL, 3.0, 3.25, '4 Hole-0.406', 'W41', 'W4'),
('02000010V', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.406', 'W5', 'W1'),
('02000141V', 100.11, 100.38, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.406', 'W5', 'W1'),
('02000299', 101.17, 101.44, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02001100', 101.17, 101.44, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02000393', 101.36, 101.63, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02000403', 94.98, 95.25, '+.010/-.100', '34', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02000187V', 94.79, 95.06, '+.010/-.100', '34', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02000772', 100.98, 101.25, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02000840', 101.36, 101.63, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02000878A', 100.73, 101.0, '+.010/-.100', '32', 6, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1"x1"x43"', '41"', NULL, NULL, 5.0, 3.938, '4 Hole-0.406', 'W5', 'W2'),
('02000883', 94.98, 95.25, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.0, 'Blank', 'Flush to Bottom', 'W3'),
('02000884', 101.36, 101.63, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.0, 'Blank', 'Flush to Bottom', 'W3'),
('02009032', 101.36, 101.63, '+.010/-.100', '33', 6, 'No', 'A1', 'A1', 'X', 'Band to Ship', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.0, 'Blank', 'Flush to Bottom', 'W3'),
('02000994V', 97.98, 98.25, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02001050', 101.36, 101.63, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02001051V', 94.98, 95.25, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02001052V', 97.98, 98.25, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02001077A', 101.36, 101.63, '+.010/-.100', '33', 6, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.0, 'Blank', 'Flush to Bottom', 'W3'),
('02001081', 101.36, 101.63, '+.010/-.100', '32', 6, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.0, 'Blank', 'Flush to Bottom', 'W3'),
('02001121A', 94.98, 95.25, '+.010/-.100', '33', 6, 'Yes', 'A1', 'A1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.0, 'Blank', NULL, 'W3'),
('02001141V', 100.73, 101.0, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.406', 'W5', 'W1'),
('02001142V', 100.73, 101.0, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.406', 'W5', 'W1'),
('02001145V', 101.36, 101.63, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W7', 'W4'),
('02001146V', 101.36, 101.63, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W7', 'W4'),
('02001472', 101.11, 101.38, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.534 Alpha', 'W6', 'W1'),
('02001473', 101.11, 101.38, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 7, 12, 84, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.938, '4 Hole-0.534 Alpha', 'W6', 'W1'),
('02001475A', 101.11, 101.38, '+.010/-.100', '32', 6, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1"x1"x43"', '41"', NULL, NULL, 5.0, 4.0, '4 Hole-0.534 Alpha', 'W6', 'W2'),
('02001480', 101.36, 101.63, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02002141V', 99.82, 101.13, '+.010/-.100', '40', 8, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 4.0, 5.25, 'Blank', NULL, 'W39'),
('02002141H', 99.82, 101.13, '+.010/-.100', '40', 8, 'Yes', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1/2"x1"x43"', '41"', NULL, NULL, 4.0, 5.25, 'Blank', NULL, 'W39'),
('02002706', 101.11, 101.38, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02002707', 95.11, 95.375, '+.010/-.100', '34', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02002708', 101.11, 101.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'W7', 'W3'),
('02002709', 95.11, 95.375, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'W7', 'W3'),
('02004899', 101.11, 101.38, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02002710', 101.11, 101.38, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02005504', 95.11, 95.38, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02002711', 95.11, 95.38, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02003253', 101.11, 101.38, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02003255', 101.11, 101.38, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02003258', 101.11, 101.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'W7', 'W3'),
('02002712', 101.11, 101.38, '+.010/-.100', '32', 6, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'W7', 'W3'),
('02005081', 101.11, 101.38, '+.010/-.100', '32', 6, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'W7', 'W3'),
('02004441', 101.36, 101.63, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02000643', 101.36, 101.63, '+.010/-.100', '35', 8, '1 web hole', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02004892', 101.11, 101.38, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02003741', 101.11, 101.38, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02005282', 101.11, 101.38, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02003263', 95.11, 95.375, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02003257', 95.11, 95.375, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02004893', 95.11, 95.375, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02004897', 101.11, 101.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'W7', 'W3'),
('02005798', 96.61, 96.88, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02005814', 94.98, 95.25, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02005815', 94.86, 95.13, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('02005383', 90.86, 91.13, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.875, '4 Hole-0.406', 'W7', 'W4'),
('02005639', 101.11, 101.38, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '4 Hole-0.406', 'W7', 'W4'),
('02006777', 101.11, 101.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'Flush to Bottom', 'W4'),
('02006700', 101.11, 101.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'Flush to Bottom', 'W4'),
('02006625', 101.11, 101.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'Flush to Bottom', 'W4'),
('02006661', 95.11, 95.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'Flush to Bottom', 'W4'),
('02006664', 95.11, 95.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'Flush to Bottom', 'W4'),
('02006681', 95.11, 95.38, '+.010/-.100', '32', 6, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 3.031, '7 Hole-0.406', 'Flush to Bottom', 'W4'),
('02007673', 101.17, 101.44, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02007320', 101.17, 101.44, '+.010/-.100', '35', 8, 'Yes', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02009016', 97.04, 97.31, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02009018', 97.04, 97.31, '+.010/-.100', '40', 8, 'Yes', 'A1', 'A1', 'X', 'Wax', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, 'Blank', 'Flush to Bottom', 'W4'),
('02005089', 101.11, 101.38, '+.010/-.100', '35', 8, 'No', 'A1', 'A1', 'X', 'Band to Ship', 6, 16, 96, '1/2"x1"x43"', '41"', NULL, NULL, 3.0, 4.0, '6 Hole-0.406', 'W7', 'W4'),
('53004039', 101.42, 101.69, '+.010/-.100', '40', 8, 'No', 'A1', 'A1', 'X', 'Wax', 7, 12, 84, '1/2"x1"x43"', '41"', NULL, NULL, 4.0, 5.0, '2 Hole', 'Flush to Bottom', 'W50'),
('TST03850', 101.0, 101.38, '+.010/-.100', '35', NULL, 'No', NULL, NULL, NULL, 'Wax', NULL, NULL, NULL, NULL, NULL, NULL, NULL, 3.0, 4.75, '5 Hole-0.438', NULL, NULL),
('TST03851', 101.0, 101.38, '+.010/-.100', '35', NULL, 'Yes', NULL, NULL, NULL, 'Wax', NULL, NULL, NULL, NULL, NULL, NULL, NULL, 3.0, 4.75, '5 Hole-0.438', NULL, NULL),
('TST03852', 101.0, 101.38, '+.010/-.100', '35', NULL, 'No', NULL, NULL, NULL, 'Wax', NULL, NULL, NULL, NULL, NULL, NULL, NULL, 3.0, 4.75, '5 Hole-0.438', NULL, NULL),
('TST03855', 101.0, 101.38, '+.010/-.100', '35', NULL, 'No', NULL, NULL, NULL, 'Band to Ship', NULL, NULL, NULL, NULL, NULL, NULL, NULL, 3.0, 4.75, 'Unknown', NULL, NULL),
('02001211A', 100.11, 100.38, '+0 / -0.093', '32', 6, 'No', 'B1', 'B1', 'X', 'Wax', 8, 12, 96, '1"x1"x43"', '41"', NULL, NULL, 5.0, 3.938, '4 Hole-0.406', 'W5', 'W2'),
('3307681', 100.48, 100.75, '+.010/-.100', NULL, 8, 'No', 'B1', 'B1', 'X', 'Band to Ship', 8, 12, 96, '1"x1"x43"', '41"', NULL, NULL, 5.0, 3.75, '5 Hole 0.406', 'Split Difference', 'W1'),
('01251004 (P)', 95.25, 95.25, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('01251003 (P)', 89.25, 89.25, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 14, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('01251005 (P)', 101.25, 101.25, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('01251022 (P)', 101.25, 101.25, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('010012 (P)', 90.69, 90.69, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 18, 108, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', 'Standing I Position', NULL),
('0100121 (P)', 96.69, 96.69, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 18, 108, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', 'Standing I Position', NULL),
('0100121C (P)', 96.69, 96.69, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 5, 10, 50, '1/2"x1"x22"', '19"', NULL, NULL, NULL, NULL, 'Unknown', 'Standing I Position', NULL),
('40102687 (P)', 101.12, 101.12, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40103283 (P)', 100.78, 100.78, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 15, 90, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', 'I Position/6H x 15W', NULL),
('10150P (P)', 101.5, 101.5, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('012050034 (P)', 101.5, 101.5, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 12, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('012050 (P)', 97.5, 97.5, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 12, 10, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('012009 (P)', 91.5, 91.5, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 12, 10, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('012057 (P)', 97.5, 97.5, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 12, 10, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('GB0151P (P)', 95.75, 95.75, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02200074 (P)', 100.63, 100.63, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('60901701X (P)', 101.39, 101.39, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('26951015P (P)', 101.48, 101.5, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('26959638P (P)', 96.38, 96.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('26951011P (P)', 101.13, 101.13, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('26951013P (P)', 101.38, 101.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('271810188 (P)', 101.88, 101.88, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', 'No Partials', NULL),
('2695102P (P)', 102.0, 102.0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('2806102P (P)', 102.0, 102.0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 7, 15, 105, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('XMB4X102P (P)', 102.0, 102.0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('2718102P (P)', 102.0, 102.0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('2919102P (P)', 102.0, 102.0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('700040071 (P)', 91.31, 91.31, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 10, 60, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('26959131 (P)', 91.31, 91.31, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('26959562P (P)', 95.62, 95.62, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('269510188 (P)', 101.88, 101.88, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', 'No Partials', NULL),
('X0200952 (P)', 95.13, 95.13, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('X0200954 (P)', 95.25, 95.25, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('012101054 (P)', 100.03, 100.03, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('010030 (P)', 93.75, 93.75, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 14, 10, 144, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('XMB00009P (P)', 101.23, 101.5, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 16, 96, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('010002H (P)', 89.62, 89.62, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 18, 108, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('6005400 (P)', 95.5, 95.5, NULL, NULL, NULL, 'Yes', NULL, NULL, NULL, 'Paint', 7, 14, 98, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('6005401 (P)', 101.5, 101.5, NULL, NULL, NULL, 'Yes', NULL, NULL, NULL, 'Paint', 7, 14, 98, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('010025H (P)', 101.62, 101.62, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 18, 108, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02006700 (P)', 101.38, 101.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 16, 96, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('33166 (P)', 101.66, 101.66, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 7, 15, 105, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('ACF51177C (P)', 101.38, 101.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 16, 96, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('A14089PNT (P)', 101.35, 101.35, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 10, 10, 100, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('X002101C (P)', 101.75, 101.75, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Paint', 6, 12, 72, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('26951015P', 101.48, 101.5, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102385G (PU)', 100.74, 100.74, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102385B (PU)', 100.74, 100.74, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102386B (PU)', 100.74, 100.74, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102386G (PU)', 100.74, 100.74, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40103246G (PU)', 100.74, 100.74, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102391X (PU)', 101.12, 101.12, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 60, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102391Y (PU)', 100.86, 100.86, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('30100466 (PU)', 100.86, 100.86, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102386M (PU)', 100.38, 100.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102385X (PU)', 100.38, 100.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40103246S (PU)', 94.38, 94.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102385Y (PU)', 94.38, 94.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102385S (PU)', 94.38, 94.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102385R (PU)', 94.38, 94.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102386S (PU)', 94.38, 94.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102386Z (PU)', 94.38, 94.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40102386Y (PU)', 94.02, 94.02, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('30101515 (PU)', 101.39, 101.39, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('30101516 (PU)', 95.03, 95.03, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('30101514 (PU)', 101.39, 101.39, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('30101519 (PU)', 95.03, 95.03, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('30101520 (PU)', 95.03, 95.03, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 96, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('30101517 (PU)', 101.39, 101.39, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 96, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('30106293G (PU)', 100.73, 100.73, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('27169503H (PU)', 95.03, 95.03, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 96, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('27161013H (PU)', 101.39, 101.39, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 96, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40103242B (PU)', 101.28, 101.28, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 96, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40105344B (PU)', 101.35, 101.35, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('14293B (PU)', 101.35, 101.35, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40103247X (PU)', 94.92, 94.92, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40103247B (PU)', 101.28, 101.28, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('2695935H (PU)', 93.5, 93.5, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('A44927-93 (PU)', 93.57, 93.57, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('A44927-99 (PU)', 99.57, 99.57, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('K95104401 (PU)', 101.36, 101.36, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('K95104402 (PU)', 101.11, 101.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('K95319306 (PU)', 101.11, 101.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('K95104403 (PU)', 95.36, 95.36, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('K95104404 (PU)', 95.11, 95.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('42575117 (PU)', 101.48, 101.48, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('3257544B (PU)', 101.48, 101.48, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('42575118 (PU)', 101.48, 101.48, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('42575120 (PU)', 95.48, 95.48, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('3257538B (PU)', 101.38, 101.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('3257538DB (PU)', 101.38, 101.38, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('3257540B (PU)', 101.13, 101.13, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('40105853 (PU)', 101.39, 101.39, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02002710 (PU)', 101.11, 101.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02002711 (PU)', 95.11, 95.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02300643 (PU)', 101.11, 101.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02300620 (PU)', 100.11, 100.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02005282 (PU)', 101.11, 101.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02001100 (PU)', 101.17, 101.17, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02007320 (PU)', 101.17, 101.17, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02300446 (PU)', 101.0, 101.0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02003741 (PU)', 101.11, 101.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02002712 (PU)', 101.11, 101.11, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 144, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02001050 (PU)', 101.36, 101.36, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02001081 (PU)', 101.36, 101.36, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 144, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('53003905 (PU)', 100.75, 100.75, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 144, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02000643 (PU)', 101.36, 101.36, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('02007673 (PU)', 101.17, 101.17, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('TST03851 (PU)', 101.0, 101.0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('6005400 (PU)', 95.23, 95.23, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 144, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('6005401 (PU)', 101.23, 101.23, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 144, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL),
('MEV-13908 (PU)', 101.36, 101.36, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 'Punch', NULL, NULL, 120, NULL, NULL, NULL, NULL, NULL, NULL, 'Unknown', NULL, NULL);

-- INSERT new WeldSpecs (using OUTER APPLY for efficient ClipId lookup)
INSERT INTO FAB.WeldSpecs (
    ProductId, CutLength, FinishedLength, Tolerance, Program, CleaningCycle,
    WebHoles, EntryKeeper, ExitKeeper, BottomKeeper, NextOp,
    BundleCol, BundleRow, BundleCount, StickSize, GroovedTimber, BandLength,
    Comments, ClipId, Author
)
SELECT
    p.ProductId,
    s.CutLength, s.FinishedLength, s.Tolerance, s.Program, s.CleaningCycle,
    s.WebHoles, s.EntryKeeper, s.ExitKeeper, s.BottomKeeper, s.NextOp,
    s.BundleCol, s.BundleRow, s.BundleCount, s.StickSize, s.GroovedTimber, s.BandLength,
    s.Comments, cl.ClipId, 'seed'
FROM #WeldSpecStaging s
JOIN FAB.Products p ON p.PartNumber = s.PartNumber
OUTER APPLY (
    SELECT TOP 1 ClipId 
    FROM FAB.catClips 
    WHERE Width = s.ClipWidth 
      AND Height = s.ClipHeight 
      AND HolePattern = s.ClipHolePattern
      AND ISNULL(HoleGage, '') = ISNULL(s.ClipHoleGage, '')
      AND ISNULL(SideGage, '') = ISNULL(s.ClipSideGage, '')
) cl
WHERE NOT EXISTS (SELECT 1 FROM FAB.WeldSpecs w WHERE w.ProductId = p.ProductId);

PRINT CONCAT('WeldSpecs inserted: ', @@ROWCOUNT);

-- UPDATE existing WeldSpecs
UPDATE w
SET 
    CutLength = s.CutLength, FinishedLength = s.FinishedLength, Tolerance = s.Tolerance,
    Program = s.Program, CleaningCycle = s.CleaningCycle, WebHoles = s.WebHoles,
    EntryKeeper = s.EntryKeeper, ExitKeeper = s.ExitKeeper, BottomKeeper = s.BottomKeeper,
    NextOp = s.NextOp, BundleCol = s.BundleCol, BundleRow = s.BundleRow, BundleCount = s.BundleCount,
    StickSize = s.StickSize, GroovedTimber = s.GroovedTimber, BandLength = s.BandLength,
    Comments = s.Comments, ClipId = cl.ClipId, Author = 'seed', DateTimeModified = GETUTCDATE()
FROM FAB.WeldSpecs w
JOIN FAB.Products p ON p.ProductId = w.ProductId
JOIN #WeldSpecStaging s ON s.PartNumber = p.PartNumber
OUTER APPLY (
    SELECT TOP 1 ClipId 
    FROM FAB.catClips 
    WHERE Width = s.ClipWidth 
      AND Height = s.ClipHeight 
      AND HolePattern = s.ClipHolePattern
      AND ISNULL(HoleGage, '') = ISNULL(s.ClipHoleGage, '')
      AND ISNULL(SideGage, '') = ISNULL(s.ClipSideGage, '')
) cl;

PRINT CONCAT('WeldSpecs updated: ', @@ROWCOUNT);

DROP TABLE #WeldSpecStaging;

COMMIT TRANSACTION;

-- ============================================================
-- VERIFICATION
-- ============================================================
PRINT '';
PRINT '=== Verification ===';

SELECT 'catCustomers' AS [Table], COUNT(*) AS [Count] FROM FAB.catCustomers
UNION ALL SELECT 'catSections', COUNT(*) FROM FAB.catSections
UNION ALL SELECT 'catGrades', COUNT(*) FROM FAB.catGrades
UNION ALL SELECT 'catClips', COUNT(*) FROM FAB.catClips
UNION ALL SELECT 'Products', COUNT(*) FROM FAB.Products
UNION ALL SELECT 'WeldSpecs', COUNT(*) FROM FAB.WeldSpecs
UNION ALL SELECT 'WeldSpecs WITH ClipId', COUNT(*) FROM FAB.WeldSpecs WHERE ClipId IS NOT NULL
UNION ALL SELECT 'WeldSpecs WITHOUT ClipId', COUNT(*) FROM FAB.WeldSpecs WHERE ClipId IS NULL;

PRINT '';
PRINT '============================================================';
PRINT 'FAB Seed Complete!';
PRINT 'Finished: ' + CONVERT(VARCHAR(30), GETDATE(), 120);
PRINT '============================================================';

END TRY
BEGIN CATCH
    IF @@TRANCOUNT > 0 ROLLBACK TRANSACTION;
    PRINT '!!! ERROR: ' + ERROR_MESSAGE() + ' (Line ' + CAST(ERROR_LINE() AS VARCHAR) + ')';
    THROW;
END CATCH;
